﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace MicroRWD.UI.MFIC
{
    public partial class WriteMifareCardMemory : Form
    {
        #region Private Properties
        private const int C_COUNT = 16;
        Mifare_MemoryPanel parent;
        private event EventHandler dataReadChanged;
        private event EventHandler Data_Read_Changed { add { dataReadChanged += value; } remove { dataReadChanged -= value; } } 
        private const byte C_LAST_BLOCK = 0x3F;
        private const byte C_LAST_KEY = 0x1F;
        private bool updateBlockFields = true;        
        private bool updateKeyFields = true;
        public byte[] Data { get; set; }
        public bool DataValid { get; set; }
        public bool Modified { get; set; }

        // Use all of the following property's accessors or manually call 
        // FireDataReadChangedEvent() if changing values directly
        #region Data Read Properties
        
        private byte block = 0x00; 
        public byte Block
        {
            get { return block; }
            set 
            { 
                block = value;
                FireDataReadChangedEvent();
            }
        }
        
        private byte keyNumber = 0x00;
        private byte KeyNumber
        {
            get { return keyNumber; }
            set 
            { 
                keyNumber = value;
                FireDataReadChangedEvent();
            }
        }

        private bool keyType = false;
        private bool KeyType
        {
            get { return keyType; }
            set 
            { 
                keyType = value;
                FireDataReadChangedEvent();
            }
        }        

        #endregion

        #endregion

        #region Constructors

        public WriteMifareCardMemory(Mifare_MemoryPanel _parent, byte _block)        
        {
            parent = _parent;
            Block = _block;
            InitializeComponent();
            Data = new byte[C_COUNT];
            DataValid = false;
            writeCardButton.Enabled = DataValid;
            Modified = false;
            Data_Read_Changed += new System.EventHandler(this.writeToMemoryPanel_Data_Read_Changed);
        }

        #endregion

        #region Private Methods
        
        private void FireDataReadChangedEvent()
        {
            if (dataReadChanged != null)
            {
                dataReadChanged(this, new EventArgs());
            }
        }

        private void refreshView()
        {
            int selectionStart;
            int selectionLength;

            if (updateBlockFields)
            {
                // Save the current text selection
                selectionStart = blockAddrHexTextBox.SelectionStart;
                selectionLength = blockAddrHexTextBox.SelectionLength;

                // Update values
                this.blockAddrHexTextBox.Text = Block.ToString("X2");

                // Restore text selection
                blockAddrHexTextBox.SelectionStart = selectionStart;
                blockAddrHexTextBox.SelectionLength = selectionLength;

                updateBlockFields = false;
            }

            if (updateKeyFields)
            {
                // Save the current text selection
                selectionStart = keyNumTextBox.SelectionStart;
                selectionLength = keyNumTextBox.SelectionLength;

                // Update values
                this.keyNumTextBox.Text = KeyNumber.ToString();
                updateKeyFields = false;

                // Restore text selection
                keyNumTextBox.SelectionStart = selectionStart;
                keyNumTextBox.SelectionLength = selectionLength;
            }

            // Update key type
            keyARadioButton.Checked = !KeyType;
            keyBRadioButton.Checked = KeyType;

            // Update Description
            descriptionTextBox.Text = parent.getDescription(Block);

            // Update Data fields
            // Save the current text selection
            selectionStart = hexTextBox.SelectionStart;
            selectionLength = hexTextBox.SelectionLength;

            if (DataValid)
            {
                hexTextBox.Text = String.Join(string.Empty, Array.ConvertAll(Data, b => b.ToString("X2") + ' '));
                hexTextBox.Enabled = true;
            }
            else
            {
                hexTextBox.Text = "XX XX XX XX XX XX XX XX XX XX XX XX XX XX XX XX ";
                hexTextBox.Enabled = false;
            }

            hexTextBox.SelectionStart = selectionStart;
            hexTextBox.SelectionLength = selectionLength;

            // Save the current text selection
            selectionStart = asciiTextBox.SelectionStart;
            selectionLength = asciiTextBox.SelectionLength;

            if (DataValid)
            {
                asciiTextBox.Text = String.Join(string.Empty, Array.ConvertAll(Data, b => ((b > 31) && (b < 127)) ? ((char)b).ToString() + ' ' : ". "));
                asciiTextBox.Enabled = true;
            }
            else
            {
                asciiTextBox.Text = ". . . . . . . . . . . . . . . . ";
                asciiTextBox.Enabled = false;
            }

            asciiTextBox.SelectionStart = selectionStart;
            asciiTextBox.SelectionLength = selectionLength;

            if (!Modified)
            {
                asciiTextBox.ForeColor = Color.Black;
                hexTextBox.ForeColor = Color.Black;
            }
            else
            {
                asciiTextBox.ForeColor = Color.Red;
                hexTextBox.ForeColor = Color.Red;
            }
        }

        private void validateValues()
        {
            // Ensure Block  Address is 
            // - between 0 - C_LAST_BLOCK
            // - less than or equal to End Address
            if (Block < 0x00)
            {
                Block = 0x00;
                updateBlockFields = true;
            }
            else if (Block > C_LAST_BLOCK)
            {
                Block = C_LAST_BLOCK;
                updateBlockFields = true;
            }

            // Ensure the RWD Key Number is within its range
            if (KeyNumber < 0x00)
            {
                KeyNumber = 0x00;
            }
            else if (KeyNumber > C_LAST_KEY)
            {
                KeyNumber = C_LAST_KEY;
            }
        }

        private bool IsHexDigit(char _c)
        {
            if (((_c >= '0') && (_c <= '9')) || ((_c >= 'A') && (_c <= 'F')) || ((_c >= 'a') && (_c <= 'f')))
            {
                return true;
            }

            return false;
        }

        private bool IsDecDigit(char _c)
        {
            if ((_c >= '0') && (_c <= '9'))
            {
                return true;
            }

            return false;
        }

        private byte HexToByte(char _c)
        {
            byte result = 0;

            if ((_c >= '0') && (_c <= '9'))
            {
                result = (byte)(_c - '0');
            }
            else if ((_c >= 'A') && (_c <= 'F'))
            {
                result = (byte)(10 + (_c - 'A'));
            }
            else if ((_c >= 'a') && (_c <= 'f'))
            {
                result = (byte)(10 + (_c - 'a'));
            }

            return result;
        }

        private void updateHexTextKeyChars(object sender, KeyPressEventArgs e, ref byte _address, ref bool _updateFlag)
        {
            System.Windows.Forms.TextBox textbox = (System.Windows.Forms.TextBox)sender;

            // Ignore non hex digit key presses
            if (IsHexDigit(e.KeyChar))
            {
                // Is cursor in front of a digit                    
                // replace digit
                // move cursor to next position (select digit if one is there)
                int cur = textbox.SelectionStart;
                if (cur == 0)
                {
                    _address &= 0x0f; // clear upper byte
                    _address |= (byte)((HexToByte(e.KeyChar) << 4) & 0xf0);
                }
                else if (cur == 1)
                {
                    _address &= 0xf0; // clear lower byte
                    _address |= HexToByte(e.KeyChar);
                }

                ++cur;
                textbox.SelectionStart = Math.Min(cur, textbox.MaxLength);
                textbox.SelectionLength = (textbox.SelectionStart < 2) ? 1 : 0;
                _updateFlag = true;

                // Refresh view
                validateValues();
                refreshView();
            }
            e.Handled = true;
        }

        private void updateDecTextKeyChars(object sender, KeyPressEventArgs e, ref byte _value, ref bool _updateFlag, bool _isStartAddr)
        {
            System.Windows.Forms.TextBox textbox = (System.Windows.Forms.TextBox)sender;

            // Ignore non dec digit key presses
            if (IsDecDigit(e.KeyChar))
            {
                // Is cursor in front of a digit                    
                // replace digit
                // move cursor to next position (select digit if one is there)
                int cur = textbox.SelectionStart;
                char[] curVal = textbox.Text.ToCharArray();
                if ((cur >= 0) && (cur <=2))
                {
                    if (cur > (curVal.Length - 1))
                    {
                        char[] temp = new char[curVal.Length + 1];
                        Array.Copy(curVal, 0, temp, 0, curVal.Length);
                        temp[curVal.Length] = e.KeyChar;

                        try
                        {
                            _value = byte.Parse(new string(temp));
                        }
                        catch (OverflowException)
                        {
                            _value = byte.MaxValue;
                        }
                    }
                    else
                    {
                        curVal[cur] = e.KeyChar;
                        _value = byte.Parse(new string(curVal));
                    }

                    ++cur;
                    textbox.SelectionStart = Math.Min(cur, textbox.MaxLength);
                    textbox.SelectionLength = (textbox.SelectionStart < 2) ? 1 : 0;
                    _updateFlag = true;
                }

                // Refresh view
                validateValues();
                refreshView();
            }
            e.Handled = true;
        }

        private void updateTextboxKeyCodes(object sender, KeyEventArgs e, ref byte _value, ref bool _updateFlag)
        {
            System.Windows.Forms.TextBox textbox = (System.Windows.Forms.TextBox)sender;
            if (e.KeyCode == Keys.Delete)
            {
                if (textbox.SelectionLength > 0)
                {
                    _value = 0;
                    textbox.SelectionStart = 0;
                    textbox.SelectionLength = 1;

                    // Refresh view
                    validateValues();
                    _updateFlag = true;
                    refreshView();
                }
                e.Handled = true;
            }
            else if (e.KeyCode == Keys.Back)
            {
                if (textbox.SelectionStart > 0)
                {
                    _value = 0;
                    textbox.SelectionStart = 0;
                    textbox.SelectionLength = 1;

                    // Refresh view
                    validateValues();
                    _updateFlag = true;
                    refreshView();
                }
                e.Handled = true;
            }
        }

        private void readData()
        {
            // Save the data held at the current block location for later use
            byte[] result = parent.readCard(Block, KeyType, KeyNumber);
            if (result.Length == 1 + C_COUNT)
            {
                // We received 1 Status byte + C_COUNT Data bytes
                Array.Copy(result, 1, Data, 0, C_COUNT);
                DataValid = true;
            }
            else
            {
                Array.Clear(Data, 0, C_COUNT);
                DataValid = false;
            }
            writeCardButton.Enabled = DataValid;
            Modified = false;
        }

        #endregion

        #region Event Handlers

        private void WriteCardMemory_Load(object sender, EventArgs e)
        {
            // Read the data for the current block
            readData();
            refreshView();
        }

        private void writeCardButton_Click(object sender, EventArgs e)
        {
            if (DataValid)
            {
                parent.writeCard(Block, KeyType, KeyNumber, Data);
                Modified = false;
                refreshView();
            }
        }

        private void closeButton_Click(object sender, EventArgs e)
        {
            Close();
        }

        private void blockAddrNextButton_Click(object sender, EventArgs e)
        {
            // Increment the block address
            if (Block< C_LAST_BLOCK)
            {
                ++Block;
                updateBlockFields = true;
            }
            validateValues();
            refreshView();
        }

        private void blockAddrPrevButton_Click(object sender, EventArgs e)
        {
            // Decrement the block address
            if (Block> 0x00)
            {
                --Block;
                updateBlockFields = true;
            }
            validateValues();
            refreshView();
        }

        private void keyNumNextButton_Click(object sender, EventArgs e)
        {
            // Increment the key number
            if (KeyNumber < C_LAST_KEY)
            {
                ++KeyNumber;
                updateKeyFields = true;
            }
            validateValues();
            refreshView();
        }

        private void keyNumPrevButton_Click(object sender, EventArgs e)
        {
            // Decrement the key number
            if (KeyNumber > 0x00)
            {
                --KeyNumber;
                updateKeyFields = true;
            }
            validateValues();
            refreshView();
        }

        private void blockAddrHexTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            byte prevBlock = block;
            updateHexTextKeyChars(sender, e, ref block, ref updateBlockFields);

            // Manually generate data read change event since the update function 
            // above does not use the accessor which handles this for you
            if (prevBlock != block)
            {
                FireDataReadChangedEvent();
            }
        }

        private void keyNumTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            byte prevKeyNumber = keyNumber;
            updateDecTextKeyChars(sender, e, ref keyNumber, ref updateKeyFields, false);

            // Manually generate data read change event since the update function 
            // above does not use the accessor which handles this for you
            if (prevKeyNumber != keyNumber)
            {
                FireDataReadChangedEvent();
            }
        }
        
        private void blockAddrHexTextBox_KeyDown(object sender, KeyEventArgs e)
        {
            byte prevBlock = block;
            updateTextboxKeyCodes(sender, e, ref block, ref updateBlockFields);
            
            // Manually generate data read change event since the update function 
            // above does not use the accessor which handles this for you
            if (prevBlock != block)
            {
                FireDataReadChangedEvent();
            }
        }

        private void keyNumTextBox_KeyDown(object sender, KeyEventArgs e)
        {
            byte prevKeyNumber = keyNumber;
            updateTextboxKeyCodes(sender, e, ref keyNumber, ref updateKeyFields);
            
            // Manually generate data read change event since the update function 
            // above does not use the accessor which handles this for you
            if (prevKeyNumber != keyNumber)
            {
                FireDataReadChangedEvent();
            }
        }

        private void keyARadioButton_CheckedChanged(object sender, EventArgs e)
        {
            KeyType = !keyARadioButton.Checked;
        }

        private void keyBRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            KeyType = keyBRadioButton.Checked;
        }

        private void writeToMemoryPanel_Data_Read_Changed(object sender, EventArgs e)
        {
            // Address changed - trigger read
            readData();

            // Refresh panel
            updateBlockFields = true;
            updateKeyFields = true;
            refreshView();
        }

        private void hexTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (C_COUNT > 0)
            {
                // Process backspace and hex digit key presses
                if (e.KeyChar == (char)8)
                {
                    // XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX
                    hexTextBox.SelectionStart = (hexTextBox.SelectionStart > 0) ? hexTextBox.SelectionStart - 1 : 0;

                    // Refresh view
                    refreshView();
                }
                else if (IsHexDigit(e.KeyChar))
                {
                    // XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX
                    int cur = hexTextBox.SelectionStart;
                    if (((cur + 1) % 3) == 0)
                    {
                        ++cur;
                    }
                    int pos = Math.Min(cur / 3, C_COUNT - 1);

                    if ((cur % 3) == 0)
                    {
                        // hi byte
                        Data[pos] = (byte)(((HexToByte(e.KeyChar) << 4) & 0xf0) | (Data[pos] & 0x0f));
                        Modified = true;
                    }
                    else
                    {
                        // lo byte
                        Data[pos] = (byte)((Data[pos] & 0xf0) | (HexToByte(e.KeyChar) & 0x0f));
                        Modified = true;
                    }

                    // XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX_XX
                    ++cur;
                    if (((cur + 1) % 3) == 0)
                    {
                        ++cur;
                    }

                    hexTextBox.SelectionStart = Math.Min(cur, hexTextBox.MaxLength - 1);

                    // Refresh view
                    refreshView();
                }
            }

            e.Handled = true;
        }

        private void asciiTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (C_COUNT > 0)
            {
                // Process backspace and ascii key presses
                if (e.KeyChar == (char)8)
                {
                    // ._._._._._._._._._._._._._._._.
                    asciiTextBox.SelectionStart = (asciiTextBox.SelectionStart > 0) ? asciiTextBox.SelectionStart - 1 : 0;

                    // Refresh view
                    refreshView();
                }
                else
                {
                    // ._._._._._._._._._._._._._._._.
                    int cur = asciiTextBox.SelectionStart;
                    if ((cur % 1) == 1)
                    {
                        ++cur;
                    }
                    int pos = Math.Min((cur + 1) / 2, C_COUNT - 1);

                    Data[pos] = (byte)e.KeyChar;
                    Modified = true;

                    cur += 2;

                    asciiTextBox.SelectionStart = Math.Min(cur, asciiTextBox.MaxLength - 1);
                    asciiTextBox.SelectionLength = 1;

                    // Refresh view
                    refreshView();
                }
            }

            e.Handled = true;
        }

        #endregion

    }
}

